
function output = panorama(imagepath)
% Generates panorama image from a set of images in imagepath directory
% Assumes images are in order and are the only files in this directory
% Note:  This skeleton code does NOT include all steps or all details required.

% Process input imagepath
files = dir(imagepath);
% Eliminate . and .. and assume everything else in directory is an input image
% Also assume that all images are color
imagelist = files(3:end);

for i=2:length(imagelist)-2
    image1 = imagelist(i).name;
    image2 = imagelist(i+1).name;
    
    % Find matching feature points between current two images using SIFT
    [~, matchIndex, loc1, loc2] = match(image1, image2);
    im1_ftr_pts = loc1(find(matchIndex > 0), 1 : 2);
    im2_ftr_pts = loc2(matchIndex(find(matchIndex > 0)), 1 : 2);
    
    % Calculate 3x3 homography matrix, H, mapping coordinates in image2 into coordinates in image1
    H = calcH(im1_ftr_pts, im2_ftr_pts);
    H_list(i) = {H};
end

% Select one input image as the reference image
% Generate new homographies that map every other image directly to the
% reference image by composing H matrices in H_list
% Save these new homographies in a list called H_map
% The homography associated with the reference image should be the identity
% matrix, created using eye(3)

image1 = imagelist(2).name;
image2 = imagelist(1).name;

% Find matching feature points between current two images using SIFT
[~, matchIndex, loc1, loc2] = match(image1, image2);
im1_ftr_pts = loc1(find(matchIndex > 0), 1 : 2);
im2_ftr_pts = loc2(matchIndex(find(matchIndex > 0)), 1 : 2);

% Calculate 3x3 homography matrix, H, mapping coordinates in image2 into coordinates in image1
H = calcH(im1_ftr_pts, im2_ftr_pts);
H_list(1) = {H};

% Map all images to the image plane of number two by computing
disp('Computing homographies to image plane 2...');
H_map(1) = {cell2mat(H_list(1))};
H_map(2) = {eye(3)};
H_map(3) = {cell2mat(H_list(2))};
H_map(4) = {cell2mat(H_list(2))*cell2mat(H_list(3))};

% Compute size of output panorama image
min_row = 1;
min_col = 1;
max_row = 0;
max_col = 0;

% for each input image
disp('Calculating canvas size...');

for i=1:length(H_map)
    cur_image = imread(imagelist(i).name);
    [rows,cols,~] = size(cur_image);
    
    % create a matrix with the coordinates of the four corners of the current image
    pt_matrix = cat(3, [1,1,1]', [1,cols,1]', [rows, 1,1]', [rows,cols,1]');
    
    % Map each of the 4 corner's coordinates into the coordinate system of the reference image
    for j=1:4
        result = H_map{i}*pt_matrix(:,:,j);
        
        min_row = floor(min(min_row, result(1)));
        min_col = floor(min(min_col, result(2)));
        max_row = ceil(max(max_row, result(1)));
        max_col = ceil(max(max_col, result(2)));
    end
    
end

% Calculate output image size
im_rows = max_row - min_row + 1;
im_cols = max_col - min_col + 1;

% Calculate offset of the upper-left corner of the reference image relative to the upper-left corner of the output image
row_offset = 1 - min_row;
col_offset = 1 - min_col;

% Initialize output image to black (0)
pan_image = zeros(im_rows, im_cols, 3);


% Perform inverse mapping for each input image
disp('Mapping final output...');
for i=1:length(H_map)
    
    cur_image = im2double(imread(imagelist(i).name));
    
    % Create a list of all pixels' coordinates in output image
    [x,y] = meshgrid(1:im_cols, 1:im_rows);
    % Create list of all row coordinates and column coordinates in separate vectors, x and y, including offset
    x = reshape(x,1,[]) - col_offset;
    y = reshape(y,1,[]) - row_offset;
    
    % Create homogeneous coordinates for each pixel in output image
    pan_pts(1,:) = y;
    pan_pts(2,:) = x;
    pan_pts(3,:) = ones(1,size(pan_pts,2));
    
    
    % Perform inverse warp to compute coordinates in current input image
    image_coords = H_map{i}\pan_pts;
    row_coords = reshape(image_coords(1,:),im_rows, im_cols);
    col_coords = reshape(image_coords(2,:),im_rows, im_cols);
    % Note:  Some values will return as NaN ("not a number") because they map to points outside the domain of the input image
    
    % Bilinear interpolate color values
    
    s = size(cur_image);
    
    weightImg = 1 : -2/(s(1)-1) : 1;
    % weight matrix: repeat of weight vector
    weightImg = 1-abs(weightImg);
    alpha = repmat(weightImg, s(2), 1);
    % Steepen the alpha gradient
    alpha = alpha*2;
    
    
    pixel_color_r = interp2(cur_image(:,:,1), col_coords, row_coords, 'linear', 0);
    pixel_color_g = interp2(cur_image(:,:,2), col_coords, row_coords, 'linear', 0);
    pixel_color_b = interp2(cur_image(:,:,3), col_coords, row_coords, 'linear', 0);
    pixel_alpha =   interp2(alpha(:,:), col_coords, row_coords, 'linear', 0);
    
    % Add to output image
    % Use horizontal gradient blending.
    
    for k=1:im_rows
        for j = 1:im_cols
            if pan_image(k,j,1) == 0 && pan_image(k,j,2) == 0 && pan_image(k,j,3) == 0
               
                pan_image(k,j,1) = pixel_color_r(k,j);
                pan_image(k,j,2)= pixel_color_g(k,j);
                pan_image(k,j,3) = pixel_color_b(k,j);
                
            end
            
            if  pixel_color_r(k,j) > 0 || pixel_color_g(k,j) > 0 || pixel_color_b(k,j) > 0
                if pixel_alpha(k,j) > 1
                    pixel_alpha(k,j) = 1;
                end
                pan_image(k,j,1) = (pixel_color_r(k,j)*pixel_alpha(k,j) + pan_image(k,j,1)*(1-pixel_alpha(k,j)));
                pan_image(k,j,2) = pixel_color_g(k,j)*pixel_alpha(k,j) + pan_image(k,j,2)*(1-pixel_alpha(k,j));
                pan_image(k,j,3) = pixel_color_b(k,j)*pixel_alpha(k,j)  + pan_image(k,j,3)*(1-pixel_alpha(k,j));
                
            end
        end
    end
    
    
end
disp('Photo-stitch successful');
output = pan_image;

end

